const moment = require('moment');
const db = require('../database/models');
const utils = require('../utils/utils');
const serviceUtils = require('./utils.js');

const {
  ActionPlan,
  UserActionPlan,
  UserAspectReinventation,
  UserNotification,
} = db;

exports.getActionPlan = async userId => {
  try {
    const currentRoute = await serviceUtils.getCurrentUserRoute(userId);
    const { id: routeId = '' } = currentRoute || {};

    const currentChallenge = await serviceUtils.getCurrentUserChallenge(userId, routeId);
    const { id: challengeId = '' } = currentChallenge || {};

    const userAspectReinventationId = await serviceUtils.getCurrentUserAspectReinventation(
      userId,
      challengeId,
      3,
    );

    const actionsPlan = await ActionPlan.findAll({
      attributes: ['id', 'name'],
      where: { challengeId },
      include: [
        {
          required: false,
          model: UserActionPlan,
          as: 'userActionPlan',
          attributes: ['id', 'active', 'startDate', 'endDate'],
          where: {
            userAspectReinventationId,
          },
        },
      ],
      order: [['userActionPlan', 'active', 'ASC']],
    });
    const auxFinalResponse = [];
    actionsPlan.forEach(h => {
      const {
        id = '', name = '', userActionPlan = [],
      } = h;
      const currentUserActionPlan = userActionPlan[0] || [];
      auxFinalResponse.push({
        id,
        name,
        active: currentUserActionPlan.active || 0,
        complete:
          currentUserActionPlan.active && moment(currentUserActionPlan.endDate) < moment() ? 1 : 0,
        startDate: currentUserActionPlan.startDate || null,
        endDate: currentUserActionPlan.endDate || null,
      });
    });

    const actionPlanActive = auxFinalResponse.filter(a => a.active === 1 && a.complete === 0)
      .sort((a, b) => new Date(a.startDate) - new Date(b.startDate));
    const actionPlanInactive = auxFinalResponse.filter(a => a.active === 0);
    const actionPlanCompleted = auxFinalResponse.filter(a => a.complete === 1);
    const finalResponse = [];
    finalResponse.push(actionPlanActive, actionPlanInactive, actionPlanCompleted);
    return {
      actionPlas: finalResponse.flat(),
      currentRoute,
      currentChallenge,
    };
  } catch (e) {
    throw e;
  }
};

exports.setActiveActionPlan = async userId => {
  try {
    const { id: routeId = '' } = await serviceUtils.getCurrentUserRoute(userId);

    const { id: challengeId = '' } = await serviceUtils.getCurrentUserChallenge(userId, routeId);

    const userAspectReinventationId = await serviceUtils.getCurrentUserAspectReinventation(
      userId,
      challengeId,
      3,
    );

    await UserAspectReinventation.update(
      { isActive: 1 },
      { where: { id: userAspectReinventationId } },
    );

    await serviceUtils.addToUserAspectReinventation(userId, challengeId, 3);

    return {};
  } catch (e) {
    throw e;
  }
};

exports.createUserActionPlan = async (
  actionPlanId,
  userId,
  startDate,
  endDate,
  active,
) => {
  try {
    const timeZone = await serviceUtils.getUserTimeZone(userId);

    const { id: routeId = '' } = await serviceUtils.getCurrentUserRoute(userId);

    const { id: challengeId = '' } = await serviceUtils.getCurrentUserChallenge(userId, routeId);

    const userAspectReinventationId = await serviceUtils.getCurrentUserAspectReinventation(
      userId,
      challengeId,
      3,
    );

    const currentActionPlan = await UserActionPlan.findOne({
      where: { actionPlanId, userAspectReinventationId },
    });

    const now = moment.tz(timeZone);

    console.log('startDate before', startDate);
    console.log('endDate before', endDate);

    startDate = moment(startDate).tz(timeZone);
    endDate = moment(endDate).tz(timeZone);

    console.log('startDate after', startDate);
    console.log('endDate after', endDate);

    // Si existe es porque ya fue activado previamente
    if (currentActionPlan) {
      startDate = currentActionPlan.startDate ? moment(currentActionPlan.startDate) : null;
    } else if (startDate.diff(now, 'days') < 0) {
      const error = {
        name: 'FECHA_INICIO_INVALIDA',
      };
      throw error;
    }

    if (endDate.diff(startDate, 'days') <= 0 || endDate.diff(startDate, 'days') > 365) {
      const error = {
        name: 'FECHA_FIN_INVALIDA',
      };
      throw error;
    }

    await utils.updateOrCreate(
      UserActionPlan,
      { actionPlanId, userAspectReinventationId },
      {
        userAspectReinventationId,
        actionPlanId,
        active: active || 1,
        startDate,
        endDate,
      },
    );

    // TODO DEFINIR A QUE HORA SE ENVIARAN ESTAS NOTIFICACIONES
    const notificationsToCreate = [];
    let notificationDateToCreate;

    const startDateNotification = startDate.clone().set({
      hour: 9, minute: 0, second: 0, millisecond: 0,
    });

    const endDateNotification = endDate.clone().set({
      hour: 9, minute: 0, second: 0, millisecond: 0,
    });

    console.log('startDateNotification', startDateNotification);
    console.log('endDateNotification', endDateNotification);

    // Notificación de inicio de plan de acción
    // Siempre se guarda porque si ya inició el plan de acción no se puede editar
    // Si la fecha de inicio es la fecha actual se programa la notificacion dentro de 30 min
    // sino se deja la fecha correspondiente
    notificationsToCreate.push({
      userId,
      actionPlanId,
      date: now.isAfter(startDateNotification.clone().add(-30, 'minutes')) ? now.clone().add(30, 'minutes') : startDateNotification,
      title: '💡 BeCrack: te felicita por crear tu plan de acción. Encuentra inspiración en tu Bitácora o el blog www.becrack.com ¡Adelante!',
      body: 'Te felicita por crear tu plan de acción. Encuentra inspiración en tu Bitácora o el blog www.becrack.com ¡Adelante!',
    });

    // Notificación de finalización de plan de acción
    notificationsToCreate.push({
      userId,
      actionPlanId,
      date: endDateNotification,
      title: '💡 BeCrack: tienes una acción que finaliza hoy en tu Plan. Celebra tus logros, o actualiza fechas. Encuentra inspiración en tu Bitácora o el blog www.becrack.com',
      body: 'Tienes una acción que finaliza hoy en tu Plan. Celebra tus logros, o actualiza fechas. Encuentra inspiración en tu Bitácora o el blog www.becrack.com',
    });

    const diffDays = endDateNotification.diff(startDateNotification, 'days');
    const halfDiffDays = Math.floor(diffDays / 2);

    // Se suman 7 días (1 semana) a la fecha de inicio del plan de acción
    notificationDateToCreate = startDateNotification.clone().add(7, 'days');

    // Mientras exista diferencia de al menos un día entre la fecha fin del plan de acción
    // y la fecha semanal
    while (endDateNotification.diff(notificationDateToCreate, 'days') > 0) {
      let title;
      let body;

      // Si ha transcurrido 50% o menos del tiempo definido para el plan de acción
      if (endDateNotification.diff(notificationDateToCreate, 'days') > halfDiffDays) {
        title = `💡 BeCrack: tienes una acción en curso en tu Plan. Van ${notificationDateToCreate.diff(startDateNotification, 'days')} días. Verifica tus avances en BeCrack o encuentra inspiración en tu Bitácora o el blog www.becrack.com`;
        body = `Tienes una acción en curso en tu Plan. Van ${notificationDateToCreate.diff(startDateNotification, 'days')} días. Verifica tus avances en BeCrack o encuentra inspiración en tu Bitácora o el blog www.becrack.com`;
      } else {
        title = `💡 BeCrack: tienes una acción pendiente en tu Plan. Faltan ${endDateNotification.diff(notificationDateToCreate, 'days')} días para terminar. Verifica avances en BeCrack. Encuentra inspiración en tu Bitácora o el blog www.becrack.com`;
        body = `Tienes una acción pendiente en tu Plan. Faltan ${endDateNotification.diff(notificationDateToCreate, 'days')} días para terminar. Verifica avances en BeCrack. Encuentra inspiración en tu Bitácora o el blog www.becrack.com`;
      }

      notificationsToCreate.push({
        userId,
        actionPlanId,
        date: notificationDateToCreate,
        title,
        body,
      });

      notificationDateToCreate = notificationDateToCreate.clone().add(7, 'days');
    }

    await UserNotification.destroy({
      where: {
        userId,
        actionPlanId,
        sent: null,
      },
    });

    if (notificationsToCreate.length) {
      await UserNotification.bulkCreate(notificationsToCreate);
    }

    return {};
  } catch (e) {
    throw e;
  }
};
