const _ = require('lodash');
const db = require('../database/models');
const serviceUtils = require('./utils.js');
const aws = require('../aws');

const {
  AspectReinventation,
  AspectReinventationType,
  Route,
  Challenge,
  UserAspectReinventation,
  User,
} = db;

exports.getAspectReinventation = async userId => {
  try {
    const currentRoute = await serviceUtils.getCurrentUserRoute(userId);
    const { id: routeId = '' } = currentRoute;

    const currentChallenge = await serviceUtils.getCurrentUserChallenge(userId, routeId);
    const { id: challengeId = '' } = currentChallenge;

    let unLockAspectReinventation = await UserAspectReinventation.findAll({
      where: { userId },
      raw: true,
      include: [
        {
          model: AspectReinventation,
          where: { challengeId },
        },
      ],
    });

    unLockAspectReinventation = unLockAspectReinventation.map(
      u => u['AspectReinventation.aspectReinventationTypeId'],
    );
    // return unLockAspectReinventation;
    let aspectsReinventation = await AspectReinventation.findAll({
      // raw: true,
      include: [
        {
          model: AspectReinventationType,
          as: 'aspectReinventationType',
        },
        {
          model: User,
          as: 'userAspectReinventations',
          attributes: ['id'],
          where: { id: userId },
          required: false,
          through: {
            attributes: ['isActive'],
          },
        },
      ],
      where: { challengeId },
    });
    aspectsReinventation = JSON.parse(JSON.stringify(aspectsReinventation));
    aspectsReinventation.forEach(a => {
      const { userAspectReinventations = [], aspectReinventationType = {} } = a;
      const { screenToNavigate = '', icon = '' } = aspectReinventationType;
      a.screenToNavigate = screenToNavigate;
      a.icon = icon;
      a.isActive = userAspectReinventations.length
        ? userAspectReinventations[0].UserAspectReinventation.isActive
        : 0;
      a.isLocked = unLockAspectReinventation.includes(a.aspectReinventationTypeId) ? 0 : 1;
      delete a.userAspectReinventations;
      delete a.aspectReinventationType;
    });

    return {
      aspectsReinventation,
      currentRoute,
      currentChallenge,
    };
  } catch (e) {
    throw e;
  }
};

exports.getResults = async userId => {

  try {
    const currentRoute = await serviceUtils.getCurrentUserRoute(userId);
    const currentChallenge = await serviceUtils.getCurrentUserChallenge(userId, currentRoute.id);

    const rawAspectsReinventation = await AspectReinventation.findAll({
      attributes: ['id', 'name', 'challengeId'],
      raw: true,
      include: [{
        required: false,
        model: User,
        as: 'userAspectReinventations',
        attributes: ['id'],
        through: {
          attributes: ['isActive'],
        },
        where: {
          id: userId,
        },
      }],
    });

    const aspectReinventations = rawAspectsReinventation.map(x => {
      delete x['userAspectReinventations.id'];
      delete x['userAspectReinventations.UserAspectReinventation.id'];
      x.isActive = x['userAspectReinventations.UserAspectReinventation.isActive'] || 0;
      x.isDone = x['userAspectReinventations.UserAspectReinventation.isActive'] || 0;
      delete x['userAspectReinventations.UserAspectReinventation.isActive'];
      return x;
    });

    const rawChallenges = await Challenge.findAll({
      attributes: ['id', 'name', 'routeId'],
      raw: true,
      include: [{
        required: false,
        model: User,
        as: 'userChallenges',
        attributes: [],
        through: {
          attributes: ['current'],
        },
        where: {
          id: userId,
        },
      }],
    });

    const challenges = rawChallenges.map(x => {
      delete x['userChallenges.UserChallenge.id'];
      x.isActive = x['userChallenges.UserChallenge.current'] || 0;
      x.isDone = _.every(aspectReinventations.filter(ar => ar.challengeId === x.id),
        ar => ar.isDone) ? 1 : 0;
      delete x['userChallenges.UserChallenge.current'];
      return x;
    });

    const rawRoutes = await Route.findAll({
      attributes: ['id', 'name', 'info'],
      raw: true,
    });

    const routes = rawRoutes.map(x => {
      x.isActive = x.id === currentRoute.id ? 1 : 0;
      x.isDone = _.every(challenges.filter(c => c.routeId === x.id),
        c => c.isDone) ? 1 : 0;
      return x;
    });

    const link = aws.getSignedUrl('files/BeCrack_Evaluar resultados.pdf');

    return {
      results: {
        routes,
        aspectReinventations: aspectReinventations.filter(ar => ar.challengeId === currentChallenge.id),
        challenges: challenges.filter(c => c.routeId === currentRoute.id),
        link,
      },
      currentRoute,
      currentChallenge,

    };
  } catch (e) {
    throw e;
  }
};
