const Aigle = require('aigle');
const _ = require('lodash');
const db = require('../database/models');
const utils = require('../utils/utils');

const {
  CardinalPower,
  Strength,
  Behavior,
  BehaviorPoint,
  Questionnarie,
  QuestionnarieBehavior,
  User,
  SuscriptionType,
  UserSuscription,
  Route,
  RouteStrength,
} = db;

/**
 * Get the necessary data to fill questionnarie
 */
exports.getInitialData = async cardinalPowerId => {
  try {
    let whereCondition;
    if (cardinalPowerId) {
      whereCondition = { id: cardinalPowerId };
    }

    const cardinalPowers = await CardinalPower.findAll({
      where: whereCondition,
      include: [
        {
          model: Strength,
          as: 'strengths',
          attributes: ['id', 'name', 'info'],
          include: [
            {
              model: Behavior,
              as: 'behaviors',
              attributes: ['id', 'name', 'info'],
            },
          ],
        },
      ],
    });

    const behaviors = await Behavior.findAll({
      attributes: ['id', 'name'],
      include: [
        {
          required: true,
          model: Strength,
          as: 'strength',
          attributes: ['id', 'name'],
          include: [
            {
              required: true,
              model: CardinalPower,
              as: 'cardinalPower',
              attributes: ['id', 'name'],
              where: whereCondition,
            },
          ],
        },
      ],
    });

    const behaviorPoints = await BehaviorPoint.findAll();
    return {
      behaviors,
      cardinalPowers,
      behaviorPoints,
    };
  } catch (e) {
    throw e;
  }
};

/**
 * Cretate a new questionnarie with its answers
 */
exports.create = async (userId, suscriptionTypeId, cardinalPowerId, answers) => {
  try {
    const findSuscriptionType = await UserSuscription.findOne({
      where: { userId, suscriptionTypeId },
    });

    if (!findSuscriptionType) {
      await UserSuscription.create({
        userId,
        suscriptionTypeId,
      });
    }

    const existQuestionnarie = await Questionnarie.findOne({
      where: {
        userId,
        suscriptionTypeId,
      },
    });

    if (existQuestionnarie) {
      console.log('Update existQuestionnarie in create');
      console.log(`userId: ${userId}, suscriptionTypeId: ${suscriptionTypeId}, cardinalPowerId: ${cardinalPowerId}`);
      await this.update(userId, suscriptionTypeId, cardinalPowerId, answers);
    } else {
      const newQuestionnarie = await Questionnarie.create({
        userId,
        suscriptionTypeId,
        cardinalPowerId: cardinalPowerId || null,
      });

      await Aigle.mapSeries(answers, async a => {
        const { behaviorId = '', behaviorPointId = '' } = a;
        await QuestionnarieBehavior.create({
          questionnarieId: newQuestionnarie.id,
          behaviorId,
          behaviorPointId,
        });
      });
    }

    return {};
  } catch (e) {
    throw e;
  }
};

/**
 * Get report related to a questionnarie of a suscriptionTypeId
 */
exports.getReport = async (userId, suscriptionTypeId) => {
  try {
    const currentQuestionnarieId = await Questionnarie.findOne({
      attributes: ['id'],
      where: { userId, suscriptionTypeId },
    });

    if (!currentQuestionnarieId) {
      console.log(`Questionnarie doesn't find userId: ${userId}, suscriptionTypeId: ${suscriptionTypeId}`);
      const error = {
        name: 'CUESTIONARIO_INEXISTENTE',
      };
      throw error;
    }

    const { id: questionnarieId = '' } = currentQuestionnarieId;

    const questionnaries = await Questionnarie.findAll({
      // logging: true,
      subQuery: false,
      where: { userId, suscriptionTypeId },
      include: [
        {
          model: User,
          as: 'user',
          // attributes: ['firstName', 'lastName'],
          include: [
            {
              model: Route,
              as: 'routes',
              through: {
                where: {
                  disabledDate: null,
                },
                attributes: [],
              },
            },
          ],
        },
        { model: SuscriptionType, as: 'suscriptionType' },
        {
          model: CardinalPower,
          as: 'cardinalPower',
        },
        {
          model: Behavior,
          as: 'behavior',
          through: {
            attributes: [],
          },
          include: [
            {
              model: Strength,
              as: 'strength',
            },
            {
              model: BehaviorPoint,
              as: 'behaviorPoint',
              through: {
                attributes: [],
                where: { questionnarieId },
              },
            },
          ],
        },
      ],
    });

    const currentQuestionnarie = questionnaries.length
      ? JSON.parse(JSON.stringify(questionnaries[0]))
      : null;

    const currentUserRoute = await User.findOne({
      where: { id: userId },
      include: [
        {
          model: Route,
          as: 'routes',
          through: {
            where: {
              disabledDate: null,
            },
            attributes: [],
          },
        },
      ],
    });

    const { routes = [] } = currentUserRoute;
    const route = routes.length ? routes[0] : null;
    const { id: routeId = '' } = route;

    const cardinalPowerResult = await CardinalPower.findAll();
    const routeStrengthResult = await RouteStrength.findAll();

    _.map(currentQuestionnarie.behavior, b => {
      b.strength.cardinalPower = cardinalPowerResult.filter(
        cp => cp.id === b.strength.cardinalPowerId,
      );
      b.strength.routeStrength = routeStrengthResult.filter(
        rs => rs.strengthId === b.strengthId && rs.routeId === routeId,
      );
    });

    const result = await utils.generatePdf(currentQuestionnarie);
    return { urlPdf: result };
  } catch (e) {
    console.log(e);
    throw e;
  }
};

/**
 * Get results related to a questionnarie of a suscriptionTypeId
 */
exports.results = async (userId, suscriptionTypeId) => {
  try {
    const existSuscription = await UserSuscription.findOne({
      where: {
        userId,
        suscriptionTypeId,
      },
    });

    if (!existSuscription) {
      console.log(`Suscription doesn't find userId: ${userId}, suscriptionTypeId: ${suscriptionTypeId}`);
      const error = {
        name: 'USUARIO_NO_POSEE_PLAN_COACHING_ACTIVO',
      };
      throw error;
    }

    const currentQuestionnarieId = await Questionnarie.findOne({
      attributes: ['id'],
      where: { userId, suscriptionTypeId },
    });

    if (!currentQuestionnarieId) {
      console.log(`Questionnarie doesn't find userId: ${userId}, suscriptionTypeId: ${suscriptionTypeId}`);
      const error = {
        name: 'CUESTIONARIO_INEXISTENTE',
      };
      throw error;
    }

    const { id: questionnarieId = '' } = currentQuestionnarieId;

    const hasCompleted = await Questionnarie.findOne({
      where: { userId, suscriptionTypeId },
      include: [
        { model: SuscriptionType, as: 'suscriptionType' },
        {
          model: CardinalPower,
          as: 'cardinalPower',
        },
        {
          model: Behavior,
          as: 'behavior',
          through: {
            attributes: [],
          },
          include: [
            {
              model: Strength,
              as: 'strength',
            },
            {
              model: BehaviorPoint,
              as: 'behaviorPoint',
              through: {
                attributes: [],
                where: { questionnarieId },
              },
            },
          ],
        },
      ],
    });

    if (!hasCompleted) {
      return {
        isCompleted: false,
        createdAt: '',
        cardinalPower: {},
        suscriptionType: {},
        resultQuestionnarie: [],
        allowUpdateAnswers: false,
      };
    }

    const { suscriptionType = {}, createdAt = '' } = hasCompleted;
    let { cardinalPower = '' } = hasCompleted;

    // let maxWeight = await BehaviorPoint.findOne({
    //   attributes: [[sequelize.fn('max', sequelize.col('weight')), 'max']],
    // });
    // maxWeight = JSON.parse(JSON.stringify(maxWeight));
    // const { max = '' } = maxWeight;

    const result = await utils.calculateQuestionnaire(hasCompleted, true);

    const createdAtTime = new Date(createdAt).getTime();
    const currentTime = new Date().getTime();

    const hoursAfterCreateQuestionnarie = Math.abs(currentTime - createdAtTime) / 3600000;

    if (parseInt(suscriptionTypeId) !== 1) {

      const cardinalPowersResult = await CardinalPower.findAll({
        attributes: ['id', 'name', 'description'],
        raw: true,
      });

      const cardinalPowersTotal = {
        1: 0,
        2: 0,
        3: 0,
        4: 0,
      };

      result.forEach(r => {
        cardinalPowersTotal[r.strength.cardinalPowerId] += parseFloat(r.valueChart);
      });

      const cardinalPowers = cardinalPowersResult.map(c => ({
        ...c,
        total: cardinalPowersTotal[c.id],
      }));

      [cardinalPower] = _.orderBy(cardinalPowers, ['total'], ['desc']);
    }

    return {
      isCompleted: !!hasCompleted,
      createdAt,
      suscriptionType,
      cardinalPower,
      resultQuestionnarie: result,
      allowUpdateAnswers: suscriptionType.id === 3 || hoursAfterCreateQuestionnarie < 24,
    };
  } catch (e) {
    throw e;
  }
};

/**
 * Update questionnarie answers
 */
exports.update = async (userId, suscriptionTypeId, cardinalPowerId, answers) => {
  try {
    const currentQuestionnarie = await Questionnarie.findOne({
      where: { userId, suscriptionTypeId },
    });

    if (!currentQuestionnarie) {
      console.log(`Questionnarie doesn't find userId: ${userId}, suscriptionTypeId: ${suscriptionTypeId}`);
      const error = {
        name: 'CUESTIONARIO_INEXISTENTE',
      };
      throw error;
    }

    const { id: currentQuestionnarieId = '', createdAt = '' } = currentQuestionnarie;

    const createdAtTime = new Date(createdAt).getTime();
    const currentTime = new Date().getTime();

    const hoursAfterCreateQuestionnarie = Math.abs(currentTime - createdAtTime) / 3600000;

    if (hoursAfterCreateQuestionnarie) {
      await Questionnarie.update(
        { cardinalPowerId: cardinalPowerId || null },
        { where: { id: currentQuestionnarieId } },
      );

      await QuestionnarieBehavior.destroy({
        where: { questionnarieId: currentQuestionnarieId },
      });

      await Aigle.mapSeries(answers, async a => {
        const { behaviorId = '', behaviorPointId = '' } = a;
        await QuestionnarieBehavior.create({
          questionnarieId: currentQuestionnarieId,
          behaviorId,
          behaviorPointId,
        });
      });
    } else {
      const error = {
        name: 'TIEMPO_EDICION_EXPIRO',
      };
      throw error;
    }

    return {};
  } catch (e) {
    throw e;
  }
};

/**
 * Envío de correo al descargar el reporte
 * */
exports.sendEmailReportDownload = async (userId) => {
  try {
    const findUserEmail = await User.findOne({
      where: { id: userId },
    });

    if (!findUserEmail) {
      const error = {
        name: 'CORREO_NO_EXISTE',
      };
      throw error;
    }

    const email = findUserEmail.email;

    const templateParams = {
      templateName: 'becrack-marketing',
      tags: ['marketing'],
      email: email.toLowerCase(),
      params: {},
    };

    const bodyMailchimp = utils.formatRequestTemplateMailchimp(templateParams);

    try {
      console.log(`Sending EMAIL DOWNLOADED REPORT to ${email}`);
      await utils.sendEmailMailchimp(bodyMailchimp);
    } catch (e) {
      console.log(`Error sending EMAIL DOWNLOADED REPORT to ${email}`);
      console.log(e);
    }

    return {};
  } catch (e) {
    throw e;
  }

}

