const jwt = require('jsonwebtoken');
const bcrypt = require('bcryptjs');
const Aigle = require('aigle');
const _ = require('lodash');
const moment = require('moment');
const config = require('../config');
const db = require('../database/models');
const utils = require('../utils/utils');
const serviceUtils = require('./utils.js');

const {
  User,
  Country,
  EducationalLevel,
  UserPreference,
  SuscriptionType,
  PreferenceType,
  Preference,
  Route,
  UserRoute,
  CoachingReason,
  Questionnarie,
  Sequelize,
  UserDeviceToken,
} = db;

const { Op } = Sequelize;

/**
 * Get the necessary data to complete the sign up
 */
exports.getRegisterInitialData = async () => {
  try {
    const countries = await Country.findAll({
      attributes: ['id', ['country2', 'name']],
    });
    const educationalLevels = await EducationalLevel.findAll();
    const preferenceTypes = await PreferenceType.findAll({});
    const preferences = await Preference.findAll({});
    const routes = await Route.findAll({});
    const coachingReasons = await CoachingReason.findAll({});

    return {
      countries,
      educationalLevels,
      preferenceTypes,
      preferences,
      routes,
      coachingReasons,
    };
  } catch (e) {
    throw e;
  }
};

/**
 * Create an new user profile
 */
exports.create = async ({
  firstName,
  lastName,
  city,
  email,
  password,
  gender,
  phone,
  countryId,
  educationalLevelId,
  preferencesId,
  routeId,
  coachingReasonId,
  initialSuscriptionTypeId,
  profileImage,
}) => {
  try {
    email = email.toLowerCase().trim();
    const findUserEmail = await User.findOne({ where: { email } });
    if (findUserEmail) {
      const error = {
        name: 'CORREO_EXISTENTE',
      };
      throw error;
    }

    const validationCodeEmail = _.times(6, () => _.random(9)).join('');
    const envExpirationCodeEmail = config.get('validationCodes.expirationTimeCodeEmail');
    const now = moment().utc();
    const expirationCodeEmail = now.add(envExpirationCodeEmail, 'minutes');

    const encryptPassword = bcrypt.hashSync(password.toString());

    const createdUserId = await User.create({
      firstName,
      lastName,
      gender,
      city,
      countryId,
      educationalLevelId,
      phone,
      email,
      password: encryptPassword,
      validationCodeEmail,
      expirationCodeEmail,
      coachingReasonId,
      initialSuscriptionTypeId,
      profileImage,
    });

    await Aigle.mapSeries(preferencesId, async p => {
      await UserPreference.create({
        userId: createdUserId.id,
        preferenceId: p,
      });
    });

    // await UserSuscription.create({
    //   userId: createdUserId.id,
    //   suscriptionTypeId: 1,
    // });

    if (routeId) {
      await UserRoute.create({
        userId: createdUserId.id,
        routeId,
        enabledDate: moment().utc(),
      });
    }

    const templateParams = {
      templateName: 'becrack-confirmaci-n-de-correo-electr-nico',
      tags: ['confirmacion-de-correo'],
      email: email.toLowerCase(),
      params: {
        code: validationCodeEmail,
      },
    };

    const bodyMailchimp = utils.formatRequestTemplateMailchimp(templateParams);

    try {
      console.log(`Sending confirmation email to ${email}`);
      await utils.sendEmailMailchimp(bodyMailchimp);
    } catch (e) {
      console.log(`Error sending confirmation email to ${email}`);
      console.log(e);
    }

    return {};
  } catch (e) {
    throw e;
  }
};

exports.login = async (email, password) => {
  try {
    let error = {};
    const result = await User.findOne({
      attributes: [
        'id',
        'firstName',
        'lastName',
        'email',
        'password',
        'profileImage',
      ],
      where: {
        email: email.toLowerCase(),
      },
      include: [
        {
          model: SuscriptionType,
          as: 'suscriptionType',
          attributes: ['id', 'name'],
          through: {
            attributes: ['createdAt'],
          },
        },
      ],
    });

    if (result) {
      const {
        id = '',
        firstName = '',
        lastName = '',
        email: emailResult = '',
        password: passwordResult = '',
        suscriptionType = [],
        profileImage = '',
      } = JSON.parse(JSON.stringify(result));

      if (bcrypt.compareSync(password, passwordResult)) {
        const _doc = {
          userId: id,
        };

        const token = jwt.sign({ _doc }, config.get('jwt.secret'), {
          expiresIn: config.get('jwt.tokenExpirationTime'),
        });

        const fullName = `${firstName} ${lastName}`.replace(/\s+/g, ' ').trim();

        const isCompletedFreeTrial = await Questionnarie.findOne({
          where: { userId: id, suscriptionTypeId: 1 },
        });

        const isCompletedAdn = await Questionnarie.findOne({
          where: { userId: id, suscriptionTypeId: 2 },
        });

        const isCompletedQuesionnarieCoaching = await Questionnarie.findOne({
          where: { userId: id, suscriptionTypeId: 3 },
        });

        const newSuscriptionTypes = _.map(suscriptionType, s => {
          const { UserSuscription = {}, ...props } = s;
          const { createdAt = '' } = UserSuscription;
          return {
            ...props,
            createdAt,
          };
        }).sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));

        const currentRoute = await serviceUtils.getCurrentUserRoute(id);
        const currentChallenge = currentRoute
          ? await serviceUtils.getCurrentUserChallenge(id, currentRoute.id) || {} : {};

        const user = {
          fullName,
          email: emailResult,
          currentSuscriptionTypeId: newSuscriptionTypes.length ? newSuscriptionTypes[0].id : null,
          currentRoute,
          currentChallengeId: currentChallenge ? currentChallenge.id : '',
          currentChallenge,
          isCompletedFreeTrial: isCompletedFreeTrial ? 1 : 0,
          isCompletedAdn: isCompletedAdn ? 1 : 0,
          isCompletedQuesionnarieCoaching: isCompletedQuesionnarieCoaching ? 1 : 0,
          profileImage,
          token,
        };
        return user;
      }

      error = {
        name: 'CREDENCIALES_INCORRECTAS',
      };
      throw error;
    }
    error = {
      name: 'USUARIO_NO_EXISTE',
    };
    throw error;
  } catch (e) {
    throw e;
  }
};

exports.confirmEmail = async (email, code) => {
  try {
    const findUserByValidationCodeEmail = await User.findOne({
      where: { email, validationCodeEmail: code },
    });

    if (!findUserByValidationCodeEmail) {
      const error = {
        name: 'CODIGO_DE_VALIDACION_DE_CORREO_INCORRECTO',
      };
      throw error;
    }

    const { expirationCodeEmail = '' } = findUserByValidationCodeEmail;
    const now = moment().utc();

    if (now > expirationCodeEmail) {
      const error = {
        name: 'CODIGO_DE_VALIDACION_EXPIRADO',
      };
      throw error;
    }

    await User.update(
      {
        confirmedEmail: now,
        validationCodeEmail: null,
        expirationCodeEmail: null,
      },
      { where: { email } },
    );

    return {};
  } catch (e) {
    throw e;
  }
};

exports.sendConfirmationEmailCode = async ({ email, userId }) => {
  try {
    let findUserEmail;
    if (email) {
      findUserEmail = await User.findOne({
        where: { email },
      });
    } else if (userId) {
      findUserEmail = await User.findOne({
        where: { id: userId },
      });
    }

    if (!findUserEmail) {
      const error = {
        name: 'CORREO_NO_EXISTE',
      };
      throw error;
    }

    const currentEmail = findUserEmail.email;

    const validationCodeEmail = _.times(6, () => _.random(9)).join('');
    const envExpirationCodeEmail = config.get('validationCodes.expirationTimeCodeEmail');
    const now = moment().utc();
    const expirationCodeEmail = now.add(envExpirationCodeEmail, 'minutes');

    await User.update(
      {
        validationCodeEmail,
        expirationCodeEmail,
      },
      { where: { email: currentEmail } },
    );

    const templateParams = {
      templateName: 'becrack-confirmaci-n-de-correo-electr-nico',
      tags: ['confirmacion-de-correo'],
      email: currentEmail,
      params: {
        code: validationCodeEmail,
      },
    };

    const bodyMailchimp = utils.formatRequestTemplateMailchimp(templateParams);

    try {
      console.log(`Sending confirmation email to ${email}`);
      await utils.sendEmailMailchimp(bodyMailchimp);
    } catch (e) {
      console.log(`Error sending confirmation email to ${email}`);
      console.log(e);
    }

    return { email: currentEmail };
  } catch (e) {
    throw e;
  }
};

exports.sendResetPasswordCode = async email => {
  try {
    const findUserEmail = await User.findOne({
      where: { email },
    });

    if (!findUserEmail) {
      const error = {
        name: 'CORREO_NO_EXISTE',
      };
      throw error;
    }

    const validationCodeResetPassword = _.times(6, () => _.random(9)).join('');

    const envExpirationTimeCodeResetPassword = config.get(
      'validationCodes.expirationTimeCodeResetPassword',
    );
    const now = moment().utc();
    const expirationCodeResetPassword = now.add(envExpirationTimeCodeResetPassword, 'minutes');

    await User.update(
      {
        validationCodeResetPassword,
        expirationCodeResetPassword,
      },
      { where: { email } },
    );

    const templateParams = {
      templateName: 'becrack-recuperaci-n-de-contrase-a',
      tags: ['recuperacion-de-contrasena'],
      email: email.toLowerCase(),
      params: {
        code: validationCodeResetPassword,
      },
    };

    const bodyMailchimp = utils.formatRequestTemplateMailchimp(templateParams);

    try {
      console.log(`Sending code recover password to ${email}`);
      await utils.sendEmailMailchimp(bodyMailchimp);
    } catch (e) {
      console.log(`Error sending code recover password to ${email}`);
      console.log(e);
    }
    return {};
  } catch (e) {
    throw e;
  }
};

exports.validateResetPasswordCode = async (email, code, password) => {
  try {
    const findUserByValidationCodeEmail = await User.findOne({
      where: { email, validationCodeResetPassword: code },
    });

    if (!findUserByValidationCodeEmail) {
      const error = {
        name: 'CODIGO_DE_VALIDACION_DE_RESETEO_PASSWORD_INCORRECTO',
      };
      throw error;
    }

    const { expirationCodeResetPassword = '' } = findUserByValidationCodeEmail;
    const now = moment().utc();

    if (now > expirationCodeResetPassword) {
      const error = {
        name: 'CODIGO_DE_VALIDACION_RESETEO_PASSWORD_EXPIRADO',
      };
      throw error;
    }

    const encryptPassword = bcrypt.hashSync(password.toString());

    await User.update(
      {
        password: encryptPassword,
        validationCodeResetPassword: null,
        expirationCodeResetPassword: null,
      },
      { where: { email } },
    );

    return {};
  } catch (e) {
    throw e;
  }
};

/**
 * Update user profile
 */
exports.update = async ({
  userId,
  firstName,
  lastName,
  city,
  email,
  gender,
  phone,
  countryId,
  educationalLevelId,
  preferencesId,
  coachingReasonId,
  profileImage,
}) => {
  try {
    email = email.toLowerCase().trim();
    const findUserEmail = await User.findOne({ where: { email, id: { [Op.ne]: userId } } });

    if (findUserEmail) {
      const error = {
        name: 'CORREO_EXISTENTE',
      };
      throw error;
    }

    await User.update({
      firstName,
      lastName,
      gender,
      city,
      countryId,
      educationalLevelId,
      phone,
      email,
      coachingReasonId,
      profileImage,
    },
    { where: { id: userId } });

    await UserPreference.destroy({ where: { userId } });
    await Aigle.mapSeries(preferencesId, async p => {
      await UserPreference.create({
        userId,
        preferenceId: p,
      });
    });

    return {};
  } catch (e) {
    throw e;
  }
};

exports.updatePassword = async ({
  userId,
  currentPassword,
  newPassword,
}) => {
  try {
    const findUser = await User.findOne({ where: { id: userId } });
    if (!findUser) {
      const error = {
        name: 'USUARIO_NO_EXISTE',
      };
      throw error;
    }
    const { password: passwordResult = '' } = findUser;

    if (!bcrypt.compareSync(currentPassword, passwordResult)) {
      const error = {
        name: 'CONTRASENAS_ACTUAL_INVALIDA',
      };
      throw error;
    }

    const encryptPassword = bcrypt.hashSync(newPassword.toString());

    await User.update({
      password: encryptPassword,
    }, { where: { id: userId } });

    return {};
  } catch (e) {
    throw e;
  }
};

/**
 * Get user profile
 */
exports.detail = async userId => {
  try {
    const result = await User.findOne({
      attributes: [
        'firstName',
        'lastName',
        'gender',
        'phone',
        'countryId',
        'city',
        'educationalLevelId',
        'email',
        'coachingReasonId',
        'profileImage',
        'confirmedEmail',
      ],
      where: {
        id: userId,
      },
      include: [
        {
          model: Preference,
          as: 'preference',
          through: {
            attributes: [],
          },
        },
      ],
    });

    if (result) {
      const {
        preference = [],
        countryId: country,
        educationalLevelId: educationalLevel,
        coachingReasonId: coachingReason,
        ...userProps
      } = JSON.parse(JSON.stringify(result));

      const preferencesIds = _.map(preference, p => p.id);

      const user = {
        ...userProps,
        preferences: preferencesIds,
        country,
        educationalLevel,
        coachingReason,
      };
      return user;
    }
    const error = {
      name: 'USUARIO_NO_EXISTE',
    };
    throw error;
  } catch (e) {
    throw e;
  }
};

/**
 * Add user device token used to send notifications
 */
exports.addDeviceToken = async ({
  userId,
  deviceToken,
  deviceOS,
  deviceVersion,
}) => {
  try {
    await UserDeviceToken.destroy({ where: { deviceToken } });
    await UserDeviceToken.create({
      userId,
      deviceToken,
      deviceOS,
      deviceVersion,
    });

    return {};
  } catch (e) {
    throw e;
  }
};

/**
 * Remove user device token
 */
exports.removeDeviceToken = async ({
  userId,
  deviceToken,
}) => {
  try {
    await UserDeviceToken.destroy({
      where: {
        userId,
        deviceToken,
      },
    });
    return {};
  } catch (e) {
    throw e;
  }
};

/**
 * Get user device tokens
 */
exports.getDeviceTokens = async userId => {
  try {
    const results = await UserDeviceToken.findAll({
      where: {
        userId,
      },
    });
    return results;
  } catch (e) {
    throw e;
  }
};

/**
 * Envíar correo lueo de que el usuario confirma su cuenta
 **/
exports.sendConfirmationRegister = async (email) => {
  try {
    const findUserEmail = await User.findOne({
      where: { email },
    });

    if (!findUserEmail) {
      const error = {
        name: 'CORREO_NO_EXISTE',
      };
      throw error;
    }

    const templateParams = {
      templateName: 'becrack-marketing',
      tags: ['marketing'],
      email: email.toLowerCase(),
      params: {},
    };

    const bodyMailchimp = utils.formatRequestTemplateMailchimp(templateParams);

    try {
      console.log(`Sending confirmation REGISTER to ${email}`);
      await utils.sendEmailMailchimp(bodyMailchimp);
    } catch (e) {
      console.log(`Error sending confirmation REGISTER to ${email}`);
      console.log(e);
    }

    return {};
  } catch (e) {
    throw e;
  }
};
