const moment = require('moment');
const Aigle = require('aigle');
const db = require('./database/models');
const constants = require('./utils/constants');
const awsActions = require('./aws');
const habitServices = require('./services/habit');

const {
  User,
  UserDeviceToken,
  UserNotification,
  Sequelize,
  Habit, UserHabit, Route, SuscriptionType, Challenge,
  AspectReinventation,
} = db;

const { Op } = Sequelize;

module.exports.sleep = ms => new Promise(resolve => setTimeout(resolve, ms));

module.exports.okResponse = (statusCode, result) => {
  const response = {
    statusCode,
    headers: {
      'Access-Control-Allow-Origin': '*',
    },
    body: JSON.stringify(result),
  };

  return response;
};

module.exports.getNextNotificationsToSend = async () => {
  try {
    const startDate = moment().utc().subtract(1, 'minutes');
    const endDate = startDate.clone().add(constants.TIME_BETWEEN_CRON, 'seconds');
    console.log(`Star date: ${startDate}`);
    console.log(`End date: ${endDate}`);

    const result = await UserNotification.findAll({
      logging: true,
      attributes: ['id', 'userId', 'date', 'title', 'body'],
      where: {
        sent: null,
        date: {
          [Op.between]: [startDate, endDate],
        },
      },
      include: [
        {
          required: true,
          model: User,
          as: 'user',
          attributes: ['id'],
          include: [
            {
              required: false,
              model: UserDeviceToken,
              as: 'devices',
              attributes: [['device_token', 'token'], ['device_os', 'os']],
            },
          ],
        },
      ],
    });

    const notifications = [];

    JSON.parse(JSON.stringify(result)).forEach(record => {
      const {
        id, title, body, user = {},
      } = record;
      const { devices = [] } = user;

      devices.forEach(device => {
        const { token, os } = device;
        notifications.push({
          id, token, os, title, body,
        });
      });
    });

    return {
      notifications,
      startDate,
      endDate,
    };

  } catch (e) {
    throw e;
  }
};

module.exports.cronSendPushNotifications = async (_event, _context, callback) => {
  try {
    const { notifications = [], startDate, endDate } = await exports.getNextNotificationsToSend();
    console.log('notifications', notifications.length);

    await Aigle.mapSeries(notifications, async notification => {
      try {

        await exports.sleep(150);
        await awsActions.sendPush(notification);

        await UserNotification.update(
          { sent: moment().utc() },
          { where: { id: notification.id } },
        );

        console.log('Enviado push notification ', notification);
      } catch (e) {
        console.log('Error enviando push notification ', notification, e);
      }
    });

    return exports.okResponse(200, { message: `Envío de push notifications ejecutado del ${startDate} al ${endDate}` });

  } catch (e) {
    return callback(null, exports.okResponse(500, {
      error: `Internal Server Error sending push notifications, ${e}`,
    }));
  }
};

module.exports.cronCreateHabitUserNotifications = async (_event, _context, callback) => {
  try {
    const users = await User.findAll({
      attributes: ['id'],
      include: [
        {
          required: true,
          model: AspectReinventation,
          as: 'userAspectReinventations',
          attributes: ['id', 'name'],
          through: {
            attributes: ['id'],
          },
          where: {
            aspectReinventationTypeId: 5,
          },
        },
        {
          required: true,
          model: SuscriptionType,
          as: 'suscriptionType',
          attributes: ['id', 'name'],
          through: {
            attributes: [],
            where: {
              suscriptionTypeId: 3,
            },
          },
        },
        {
          required: true,
          model: Route,
          as: 'routes',
          attributes: ['id', 'name'],
          through: {
            attributes: [],
            where: {
              disabledDate: null,
            },
          },
        },
        {
          required: true,
          model: Challenge,
          as: 'userChallenges',
          attributes: ['id', 'name', 'routeId'],
          through: {
            attributes: [],
            where: {
              current: 1,
            },
          },
          include: [
            {
              required: true,
              model: AspectReinventation,
              as: 'aspectReinventation',
              attributes: ['id', 'name'],
              where: {
                aspectReinventationTypeId: 5,
              },
            },
          ],
        },
      ],
    });

    console.log('usersIds', users.map(u => u.id));

    const usersData = users.map(user => {
      const { routes = [], userChallenges = [], userAspectReinventations = [] } = user;
      if (routes.length === 0) {
        return null;
      }

      const currentRoute = routes[0];

      const currentChallenge = userChallenges.find(c => c.routeId === currentRoute.id);

      if (!currentChallenge) {
        return null;
      }

      if (currentChallenge.aspectReinventation.length === 0) {
        return null;
      }

      const habitAspectReinventation = currentChallenge.aspectReinventation[0];

      const aspectReinventationId = habitAspectReinventation.id;

      const userAspectReinventation = userAspectReinventations
        .find(item => item.id === aspectReinventationId);

      if (!userAspectReinventation || !userAspectReinventation.UserAspectReinventation) {
        return null;
      }

      return ({
        userId: user.id,
        routeId: currentRoute.id,
        challengeId: currentChallenge.id,
        aspectReinventationId,
        userAspectReinventationId: userAspectReinventation.UserAspectReinventation.id,
      });
    }).filter(item => Boolean(item));

    console.log('usersData', usersData);

    await Aigle.mapSeries(usersData, async user => {
      try {
        const habits = await Habit.findAll({
          attributes: ['id', 'name'],
          where: { challengeId: user.challengeId },
          include: [
            {
              required: true,
              model: UserHabit,
              as: 'userHabit',
              attributes: ['id'],
              where: {
                userAspectReinventationId: user.userAspectReinventationId,
                completedDate: null,
              },
            },
          ],
        });

        await Aigle.mapSeries(habits, async habit => {
          const hasHabitNotificationsInTheFuture = await UserNotification.findOne({
            where: {
              userId: user.userId,
              habitId: habit.id,
              date: { [Op.gt]: moment().utc() },
            },
            raw: true,
          });

          if (!hasHabitNotificationsInTheFuture) {
            await habitServices.createHabitWeekNotifications({
              userId: user.userId,
              habitId: habit.id,
            });

          } else {
            console.log(`${user.userId} already has habits notifications in the future habitId: ${habit.id}`);
          }
        });

      } catch (e) {
        console.log(`Error creando notificaciones de hábitos para userId  ${user.userId}`, e);
      }
    });

    return exports.okResponse(200, { message: 'Notificaciones de hábitos creadas de push notifications' });

  } catch (e) {
    return callback(null, exports.okResponse(500, {
      error: `Internal Server Error creando notificaciones de hábitos, ${e}`,
    }));
  }
};

/* module.exports.cancelOrResuscribedApple = async _event => {
  try {

    const { signedPayload = '' } = _event;
    const splitInfoBase64 = signedPayload.split('.')[1];
    let notificationInfoJson = Buffer.from(splitInfoBase64, 'base64');
    notificationInfoJson = JSON.parse(notificationInfoJson.toString('ascii'));

    const { notificationType = '', data = '' } = notificationInfoJson;
    const {
      signedTransactionInfo: signedTransactionInfoBase64 = '',
      signedRenewalInfo: signedRenewalInfoBase64 = '',
    } = data;

    const splitSignedTransactionInfoBase64 = signedTransactionInfoBase64.split('.')[1];
    let signedTransactionInfoJson = Buffer.from(splitSignedTransactionInfoBase64, 'base64');
    signedTransactionInfoJson = JSON.parse(signedTransactionInfoJson.toString('ascii'));

    const splitSignedRenewalInfo = signedRenewalInfoBase64.split('.')[1];
    let signedRenewalInfoJson = Buffer.from(splitSignedRenewalInfo, 'base64');
    signedRenewalInfoJson = JSON.parse(signedRenewalInfoJson.toString('ascii'));

    return {
      notificationType,
      signedTransactionInfoJson,
      signedRenewalInfoJson,
    };

  } catch (e) {
    throw e;
  }
}; */
