const express = require('express');
const cookieParser = require('cookie-parser');
const cors = require('cors');
const helmet = require('helmet');
const createError = require('http-errors');
const expressJwt = require('express-jwt');
const serverless = require('serverless-http');
const utils = require('./utils/utils');
const config = require('./config');
/* ROUTERS */
const userRouter = require('./routes/user');
const questionnarieRouter = require('./routes/questionnarie');
const challengeRouter = require('./routes/challenge');
const habitRouter = require('./routes/habit');
const aspectReinventationRouter = require('./routes/aspectReinventation');
const actionPlanRouter = require('./routes/actionPlan');
const powerfulQuestionRouter = require('./routes/powerfulQuestion');
const faqRouter = require('./routes/faq');
const userStrengthRouter = require('./routes/userStrength');
const userStrengthCoaching = require('./routes/userStrengthCoaching');
const routeRouter = require('./routes/routes');
const answerRouter = require('./routes/answer');
const designStategyRouter = require('./routes/designStrategy');
const suscriptionRouter = require('./routes/suscription');
const giftRouter = require('./routes/gift');
const cardinalPowerRouter = require('./routes/cardinal-power');
const { planCoachingMiddleware } = require('./middleware');

const app = express();

const whitelist = config.get('cors.urls').split(',');
const corsOptions = {
  origin: (origin, callback) => {
    if (whitelist.indexOf(origin) !== -1 || whitelist.indexOf('*') !== -1) {
      callback(null, true);
    } else {
      console.log(`Error cors origin: ${origin}`);
      const error = createError(403, 'Not authorized');
      callback(error);
    }
  },
};
app.options('*', cors(corsOptions));
app.use(cors(corsOptions));
app.use(helmet());
app.use(express.urlencoded({ extended: false }));
app.use(express.json());
app.use(cookieParser());

const unlessPathsJWT = [/.*\/api\/([A-Za-z0-9-]+)\/external/];
// const unlessPathsJWT = [];

app.use(
  expressJwt({ secret: config.get('jwt.secret'), ignoreExpiration: true }).unless({
    path: unlessPathsJWT,
  }),
  (err, req, res, next) => {
    if (err) {
      console.log(err);
      return utils.errorResponse(res, 'TOKEN_INVALIDO', err);
    }
    next();
  },
);

app.use('/api/users', userRouter);
app.use('/api/faqs', faqRouter);
app.use('/api/answers', answerRouter);
app.use('/api/questionnaries', questionnarieRouter);
app.use('/api/user-strengths', userStrengthRouter);
app.use('/api/routes', routeRouter);
app.use('/api/cardinal-powers', cardinalPowerRouter);
app.use('/api/suscriptions', suscriptionRouter);
app.use('/api/gifts', giftRouter);

// Servicios usados solo para Plan Coaching
app.use('/api/challenges', planCoachingMiddleware, challengeRouter);
app.use('/api/habits', planCoachingMiddleware, habitRouter);
app.use('/api/aspects-reinventation', planCoachingMiddleware, aspectReinventationRouter);
app.use('/api/actions-plan', planCoachingMiddleware, actionPlanRouter);
app.use('/api/powerful-questions', planCoachingMiddleware, powerfulQuestionRouter);
app.use('/api/design-strategy', planCoachingMiddleware, designStategyRouter);
app.use('/api/coaching/user-strengths', planCoachingMiddleware, userStrengthCoaching);

// catch 404 and forward to error handler
app.use((req, res, next) => {
  next(createError(404));
});

module.exports.handler = serverless(app);
