"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const elliptic_1 = require("elliptic");
// @ts-ignore
const asn1 = require("asn1.js");
const BN = require('bn.js');
const ECPrivateKeyASN = asn1.define('ECPrivateKey', function () {
    // @ts-ignore
    const self = this;
    self.seq().obj(self.key('version').int(), self.key('privateKey').octstr(), self.key('parameters').explicit(0).objid().optional(), self.key('publicKey').explicit(1).bitstr().optional());
});
const SubjectPublicKeyInfoASN = asn1.define('SubjectPublicKeyInfo', function () {
    // @ts-ignore
    const self = this;
    self.seq().obj(self.key('algorithm').seq().obj(self.key("id").objid(), self.key("curve").objid()), self.key('pub').bitstr());
});
const curves = {
    secp256k1: {
        curveParameters: [1, 3, 132, 0, 10],
        privatePEMOptions: { label: 'EC PRIVATE KEY' },
        publicPEMOptions: { label: 'PUBLIC KEY' },
        curve: new elliptic_1.ec('secp256k1')
    }
};
class KeyEncoder {
    constructor(options) {
        if (typeof options === 'string') {
            if (options !== 'secp256k1') {
                throw new Error('Unknown curve ' + options);
            }
            options = curves[options];
        }
        this.options = options;
        this.algorithmID = [1, 2, 840, 10045, 2, 1];
    }
    privateKeyObject(rawPrivateKey, rawPublicKey) {
        const privateKeyObject = {
            version: new BN(1),
            privateKey: Buffer.from(rawPrivateKey, 'hex'),
            parameters: this.options.curveParameters
        };
        if (rawPublicKey) {
            privateKeyObject.publicKey = {
                unused: 0,
                data: Buffer.from(rawPublicKey, 'hex')
            };
        }
        return privateKeyObject;
    }
    publicKeyObject(rawPublicKey) {
        return {
            algorithm: {
                id: this.algorithmID,
                curve: this.options.curveParameters
            },
            pub: {
                unused: 0,
                data: Buffer.from(rawPublicKey, 'hex')
            }
        };
    }
    encodePrivate(privateKey, originalFormat, destinationFormat) {
        let privateKeyObject;
        /* Parse the incoming private key and convert it to a private key object */
        if (originalFormat === 'raw') {
            if (typeof privateKey !== 'string') {
                throw 'private key must be a string';
            }
            let keyPair = this.options.curve.keyFromPrivate(privateKey, 'hex');
            let rawPublicKey = keyPair.getPublic('hex');
            privateKeyObject = this.privateKeyObject(privateKey, rawPublicKey);
        }
        else if (originalFormat === 'der') {
            if (typeof privateKey !== 'string') {
                // do nothing
            }
            else if (typeof privateKey === 'string') {
                privateKey = Buffer.from(privateKey, 'hex');
            }
            else {
                throw 'private key must be a buffer or a string';
            }
            privateKeyObject = ECPrivateKeyASN.decode(privateKey, 'der');
        }
        else if (originalFormat === 'pem') {
            if (typeof privateKey !== 'string') {
                throw 'private key must be a string';
            }
            privateKeyObject = ECPrivateKeyASN.decode(privateKey, 'pem', this.options.privatePEMOptions);
        }
        else {
            throw 'invalid private key format';
        }
        /* Export the private key object to the desired format */
        if (destinationFormat === 'raw') {
            return privateKeyObject.privateKey.toString('hex');
        }
        else if (destinationFormat === 'der') {
            return ECPrivateKeyASN.encode(privateKeyObject, 'der').toString('hex');
        }
        else if (destinationFormat === 'pem') {
            return ECPrivateKeyASN.encode(privateKeyObject, 'pem', this.options.privatePEMOptions);
        }
        else {
            throw 'invalid destination format for private key';
        }
    }
    encodePublic(publicKey, originalFormat, destinationFormat) {
        let publicKeyObject;
        /* Parse the incoming public key and convert it to a public key object */
        if (originalFormat === 'raw') {
            if (typeof publicKey !== 'string') {
                throw 'public key must be a string';
            }
            publicKeyObject = this.publicKeyObject(publicKey);
        }
        else if (originalFormat === 'der') {
            if (typeof publicKey !== 'string') {
                // do nothing
            }
            else if (typeof publicKey === 'string') {
                publicKey = Buffer.from(publicKey, 'hex');
            }
            else {
                throw 'public key must be a buffer or a string';
            }
            publicKeyObject = SubjectPublicKeyInfoASN.decode(publicKey, 'der');
        }
        else if (originalFormat === 'pem') {
            if (typeof publicKey !== 'string') {
                throw 'public key must be a string';
            }
            publicKeyObject = SubjectPublicKeyInfoASN.decode(publicKey, 'pem', this.options.publicPEMOptions);
        }
        else {
            throw 'invalid public key format';
        }
        /* Export the private key object to the desired format */
        if (destinationFormat === 'raw') {
            return publicKeyObject.pub.data.toString('hex');
        }
        else if (destinationFormat === 'der') {
            return SubjectPublicKeyInfoASN.encode(publicKeyObject, 'der').toString('hex');
        }
        else if (destinationFormat === 'pem') {
            return SubjectPublicKeyInfoASN.encode(publicKeyObject, 'pem', this.options.publicPEMOptions);
        }
        else {
            throw 'invalid destination format for public key';
        }
    }
}
exports.default = KeyEncoder;
KeyEncoder.ECPrivateKeyASN = ECPrivateKeyASN;
KeyEncoder.SubjectPublicKeyInfoASN = SubjectPublicKeyInfoASN;
//# sourceMappingURL=key-encoder.js.map