const Aigle = require('aigle');
const _ = require('lodash');
const db = require('../database/models');
const utils = require('../utils/utils');
const serviceUtils = require('./utils.js');

const {
  UserStrength,
  Challenge,
  Strength,
  User,
  UserAspectReinventation,
  RouteStrength,
  Questionnarie,
  QuestionnarieBehavior,
  BehaviorPoint,
  Behavior,
} = db;

exports.getUserStrengths = async userId => {
  try {
    const currentRoute = await serviceUtils.getCurrentUserRoute(userId);
    const { id: routeId = '' } = currentRoute;

    const currentChallenge = await serviceUtils.getCurrentUserChallenge(userId, routeId);
    const { id: challengeId = '' } = currentChallenge;

    const strengths = await Strength.findAll({
      attributes: ['id', 'name', 'info'],
      include: [
        {
          model: Challenge,
          as: 'challenge',
          where: { id: challengeId },
          attributes: [],
        },
        {
          model: User,
          as: 'user',
          required: false,
          where: { id: userId },
          attributes: ['id'],
          through: { attributes: ['value'] },
        },
      ],
    });

    const strengthIds = strengths.map(s => s.id);

    const currentCoachingQuestionnarie = await Questionnarie.findOne({
      attributes: ['id'],
      where: { userId, suscriptionTypeId: 3 },
    });

    let currentCoachingQuestionnarieResponses = [];

    if (currentCoachingQuestionnarie) {
      const { id: coachingQuestionnarieId = '' } = currentCoachingQuestionnarie;
      currentCoachingQuestionnarieResponses = await QuestionnarieBehavior.findAll({
        attributes: ['id'],
        where: { questionnarieId: coachingQuestionnarieId },
        include: [
          {
            model: Behavior,
            as: 'Behavior',
            attributes: ['id', 'strengthId'],
            where: { strengthId: strengthIds },
          },
          {
            model: BehaviorPoint,
            as: 'BehaviorPoint',
            attributes: ['weight'],
          },
        ],
        raw: true,
      });
    }

    const finalResponse = [];
    let baseBehaviors = [];
    let acum = 0;
    strengths.forEach(s => {
      const {
        id = '', name = '', info = '',
      } = s;
      baseBehaviors = currentCoachingQuestionnarieResponses.filter(f => f['Behavior.strengthId'] === id);
      acum = _.sumBy(baseBehaviors, o => parseFloat(o['BehaviorPoint.weight']));
      finalResponse.push({
        id,
        name,
        info,
        value: baseBehaviors.length > 0 ? (acum / baseBehaviors.length).toFixed(2) : 0,
      });
    });

    return {
      questions: finalResponse,
      currentChallenge,
      currentRoute,
    };
  } catch (e) {
    throw e;
  }
};

exports.updateValuesUserStrengths = async (strengthIds, userId) => {
  try {
    await Aigle.mapSeries(strengthIds, async s => {
      const { strengthId = '', value = '' } = s;
      await utils.updateOrCreate(
        UserStrength,
        { strengthId, userId },
        { strengthId, userId, value: value || 0 },
      );
    });
    return {};
  } catch (e) {
    throw e;
  }
};

exports.setActiveUserStrengths = async userId => {
  try {
    const { id: routeId = '' } = await serviceUtils.getCurrentUserRoute(userId);

    const { id: challengeId = '' } = await serviceUtils.getCurrentUserChallenge(userId, routeId);

    const userAspectReinventationId = await serviceUtils.getCurrentUserAspectReinventation(
      userId,
      challengeId,
      4,
    );

    await UserAspectReinventation.update(
      { isActive: 1 },
      { where: { id: userAspectReinventationId } },
    );

    await serviceUtils.addToUserAspectReinventation(userId, challengeId, 4);
    return {};
  } catch (e) {
    throw e;
  }
};

exports.getMobileText = async (userId, strengthId) => {
  try {
    const { id: routeId = '' } = await serviceUtils.getCurrentUserRoute(userId);
    const currentStrength = await RouteStrength.findOne({ where: { routeId, strengthId } });
    const detailStrength = await Strength.findOne({ where: { id: strengthId } });
    const { mobilePath = '' } = currentStrength;
    const mobileText = utils.getMobileValue(mobilePath);
    const { description = [], listOne = [], listTwo = [] } = mobileText;
    const { name, icon } = detailStrength;
    return {
      name, icon, description, listOne, listTwo,
    };
  } catch (e) {
    throw e;
  }
};
